﻿using System;
using System.Threading.Tasks;
using SoapConsumer.Infrastructure.CQRS.Commands.Generic;
using SoapConsumer.Infrastructure.CQRS.Events;
using SoapConsumer.Infrastructure.CQRS.Events.Generic;

namespace SoapConsumer.Infrastructure.CQRS.Commands
{
    public class CommandBus : ICommandBus
    {
        private readonly Func<Type, IHandleCommand> _handlersFactory;
        private readonly IEventBus _eventBus;

        public CommandBus(Func<Type, IHandleCommand> handlersFactory, IEventBus eventBus)
        {
            _handlersFactory = handlersFactory;
            _eventBus = eventBus;
        }

        public async Task SendAsync<TCommand>(TCommand command)
            where TCommand : ICommand
        {
            var commandBeforeEvent = new BeforeCommandExecutionEvent<TCommand> { Command = command };
            await _eventBus.PublishAsync(commandBeforeEvent);
            if (commandBeforeEvent.Abort)
            {
                return;
            }

            var handler = (IHandleCommand<TCommand>)this._handlersFactory(typeof(TCommand));
            if (handler == null)
            {
                throw new InvalidOperationException($"Handler for {typeof(TCommand)} probably was not defined.");
            }

            await handler.ExecuteAsync(command);

            var commandAfterEvent = new AfterCommandExecutionEvent<TCommand> { Command = command };
            await _eventBus.PublishAsync(commandAfterEvent);
        }
    }
}