﻿using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Reflection;
using System.Threading.Tasks;
using System.Xml;
using Microsoft.Extensions.Configuration;
using SoapConsumer.Infrastructure.Helpers;
using SoapConsumer.Infrastructure.Proxy.Abstract;
using SoapConsumer.Infrastructure.Proxy.Attributes;
using SoapConsumer.Infrastructure.Proxy.Builders.Abstract;
using SoapConsumer.Infrastructure.Transformers.Abstract;

namespace SoapConsumer.Infrastructure.Proxy
{
    public class WebServiceProxy : IWebServiceProxy
    {
        private readonly IPipelineTransformer _pipelineTransformer;
        private readonly ISoapXmlBuilder _builder;
        private readonly IConfiguration _configuration;

        public WebServiceProxy(IPipelineTransformer pipelineTransformer,
            ISoapXmlBuilder builder ,IConfiguration configuration)
        {
            _pipelineTransformer = pipelineTransformer;
            _builder = builder;
            _configuration = configuration;
        }

        public async Task<TModel> ExecuteAsync<TModel, TParams>(TParams parameters)
        {
            if (!(parameters?.GetType().GetCustomAttributes(typeof(SoapAttribute)).FirstOrDefault() is SoapAttribute
                soapAttribute))
                throw new InvalidOperationException("Object doesn't have soap attribute.");

            var host = _configuration.GetSection(soapAttribute.Host ?? "WebService:Hosts:0:DefaultHost")
                .GetValue();
            //var endpoint = _configuration.GetSection(soapAttribute.Endpoint).GetValue();
            //var operation = _configuration.GetSection(soapAttribute.Operation).GetValue(); 
            var endpoint = soapAttribute.Endpoint;
            var operation = soapAttribute.Operation;

            var request = CreateWebRequest(host, endpoint, operation);
            var soapEnvelopeXml = new XmlDocument();

            var xmlEnvelope = _builder.Build(parameters, soapAttribute.Namespace);
            soapEnvelopeXml.LoadXml(xmlEnvelope);

            using (var stream = request.GetRequestStream())
            {
                soapEnvelopeXml.Save(stream);
            }

            return await GetResponseAsync<TModel>(request);
        }

        private async Task<TModel> GetResponseAsync<TModel>(HttpWebRequest request)
        {
            using (var response = await request.GetResponseAsync())
            {
                using (var rd = new StreamReader(response.GetResponseStream()))
                {
                    return _pipelineTransformer.Transform<TModel>(rd.ReadToEnd());
                }
            }
        }

        private HttpWebRequest CreateWebRequest(string host, string endpoint, string operation)
        {
            var webRequest = (HttpWebRequest)WebRequest.Create($@"{host}{endpoint}/{operation}");
            webRequest.Headers.Add(@"SOAP:Action");
            webRequest.ContentType = "text/xml;charset=\"utf-8\"";
            webRequest.Accept = "text/xml";
            webRequest.Method = "POST";
            return webRequest;
        }
    }
}