﻿using System;
using System.Collections.Generic;
using System.Linq;
using Autofac;
using SoapConsumer.Infrastructure.CQRS.Commands;
using SoapConsumer.Infrastructure.CQRS.Commands.Generic;
using SoapConsumer.Infrastructure.CQRS.Events;
using SoapConsumer.Infrastructure.CQRS.Events.Generic;
using SoapConsumer.Infrastructure.CQRS.Queries;
using SoapConsumer.Infrastructure.CQRS.Queries.Generic;

namespace SoapConsumer.Infrastructure.Modules
{
    public class CQRSModule : Module
    {
        protected override void Load(ContainerBuilder builder)
        {
            base.Load(builder);

            RegisterEventsBus(builder);
            RegisterCommandBus(builder);
            RegisterQueryBus(builder);
        }

        public void RegisterCommandBus(ContainerBuilder builder)
        {
            base.Load(builder);

            builder.RegisterAssemblyTypes(ThisAssembly)
                .Where(x => x.IsAssignableTo<IHandleCommand>())
                .AsImplementedInterfaces();

            builder.Register<Func<Type, IHandleCommand>>(c =>
            {
                var ctx = c.Resolve<IComponentContext>();

                return t =>
                {
                    var handlerType = typeof(IHandleCommand<>).MakeGenericType(t);
                    return (IHandleCommand)ctx.Resolve(handlerType);
                };
            });

            builder.RegisterType<CommandBus>().AsImplementedInterfaces();
        }

        protected void RegisterEventsBus(ContainerBuilder builder)
        {
            builder.RegisterAssemblyTypes(ThisAssembly)
            .Where(x => x.IsAssignableTo<IHandleEvent>())
            .AsImplementedInterfaces();

            builder.Register<Func<Type, IEnumerable<IHandleEvent>>>(c =>
            {
                var ctx = c.Resolve<IComponentContext>();
                return t =>
                {
                    var handlerType = typeof(IHandleEvent<>).MakeGenericType(t);
                    var handlersCollectionType = typeof(IEnumerable<>).MakeGenericType(handlerType);
                    return (IEnumerable<IHandleEvent>)ctx.Resolve(handlersCollectionType);
                };
            });

            builder.RegisterType<EventBus>().AsImplementedInterfaces();
        }

        public void RegisterQueryBus(ContainerBuilder builder)
        {
            base.Load(builder);

            builder.RegisterAssemblyTypes(ThisAssembly)
                .Where(x => x.IsAssignableTo<IHandleQuery>())
                .AsImplementedInterfaces();

            builder.Register<Func<Type, IHandleQuery>>(c =>
            {
                var ctx = c.Resolve<IComponentContext>();

                return t =>
                {
                    var queryInterfaceType = t.GetInterfaces().First(i => typeof(IQuery).IsAssignableFrom(i));
                    var handlerType = typeof(IHandleQuery<,>).MakeGenericType(t, queryInterfaceType.GenericTypeArguments.FirstOrDefault());
                    return (IHandleQuery)ctx.Resolve(handlerType);
                };
            });

            builder.RegisterType<QueryBus>().AsImplementedInterfaces();
        }
    }
}